-- Module with base functions similar in all devices
local m = {}

local IdTable
local CoTable
local device
local channels
local useUrns = false

function m.Init(_IdTable, _CoTable, _device, _channels)
  useUrns  = false
  IdTable  = _IdTable
  CoTable  = _CoTable
  device   = _device
  channels = _channels
end

-- Table containing data point size in bit for each DPT main number from 1 to 31 and 232
local DptSizeFromMain = {
  -- 1, 2, 3, 4, 5, 6,  7,  8,  9, 10, 11, 12, 13, 14, 15,  16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31
     1, 2, 4, 8, 8, 8, 16, 16, 16, 24, 24, 32, 32, 32, 32, 112,  8,  8, 64,  8,  8, 16,  2,  0,  8,  8, 32,  0, 64, 24,  3
}
DptSizeFromMain[232] = 24

function m.DPT(mainNumber, subNumber, dataSize)
  if not mainNumber then
    return { 0, 0, 0 }
  end
  dataSize = DptSizeFromMain[mainNumber] or dataSize or 0
  return { dataSize, mainNumber, subNumber or 0 }
end

--[[
  firstId  = Range of object IDs for this channel
  firstCo  = Range of communication objects for this channel
  chnCount = Number of channels of this channel type
  dpPerChn = How many data points are in a channel of this type
  coPerDp  = How many communication objects are linked to one data point
  idOffset = Object ID offset between two channels, usually dpPerChn + 1
  chnType  = Name of the channel type
  dpts     = Optional table of data point types for each data point with different size
]]
function m.Channel(firstId, firstCo, chnCount, dpPerChn, coPerDp, idOffset, chnType, dpts)
  local lastId = firstId + chnCount * idOffset - 1
  local lastCo = firstCo + chnCount * dpPerChn * coPerDp - 1
  return { firstId = firstId, lastId = lastId, firstCo = firstCo, lastCo = lastCo,
           dpPerChn = dpPerChn, coPerDp = coPerDp, idOffset = idOffset, chnType = chnType, dpts = dpts }
end

-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- [local] Get object ID and data point type from communication object number.
function m.GetIDandDPTFromCO(co)
  local id = 0
  local dpt = m.DPT()
  if co >= device.firstCo and co <= device.lastCo then
    id = CoTable[co] and CoTable[co][1] or id
    dpt = CoTable[co] and CoTable[co][2] or dpt
    return id, dpt[1], dpt[2], dpt[3]
  end

  local channel
  for _, v in pairs(channels) do
    if co >= v.firstCo and co <= v.lastCo then
      channel = v
      break
    end
  end

  if not channel then
    return id, dpt[1], dpt[2], dpt[3]
  end

  local coInRange = co - channel.firstCo
  local coInterval = channel.dpPerChn * channel.coPerDp
  local channelIndex = math.floor(coInRange / coInterval)
  local coIndex = coInRange % coInterval
  local dpIndex = math.floor(coIndex / channel.coPerDp)
  id = channel.firstId + channelIndex * channel.idOffset + dpIndex + 1
  if channel.dpts and channel.dpts[dpIndex + 1] and channel.dpts[dpIndex + 1][1] then
    dpt = channel.dpts[dpIndex + 1][1]
  end
  return id, dpt[1], dpt[2], dpt[3]
end

-- define a global function call
local func_GetIDandDPTFromCO = function(co) return m.GetIDandDPTFromCO(co) end
-- overwrite the local function call
function m.SetFunction_GetIDandDPTFromCO(f)
  if type(f)=="function" then
    func_GetIDandDPTFromCO = f
  end
end
-- [global] Get object ID and data point type from communication object number.
function GetIDandDPTFromCO(co)
  return func_GetIDandDPTFromCO(co)
end
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- [local] Get object ID and data point type from communication object number by specific size.
function m.GetIDandDPTFromCOBySize(co, size)
  local id = 0
  local dpt = m.DPT()
  if co >= device.firstCo and co <= device.lastCo then
    id = CoTable[co] and CoTable[co][1] or id
    dpt = CoTable[co] and CoTable[co][2] or dpt
    return id, dpt[1], dpt[2], dpt[3]
  end

  local channel
  for _, v in pairs(channels) do
    if co >= v.firstCo and co <= v.lastCo then
      channel = v
      break
    end
  end

  if not channel then
    return id, dpt[1], dpt[2], dpt[3]
  end

  local coInRange = co - channel.firstCo
  local coInterval = channel.dpPerChn * channel.coPerDp
  local channelIndex = math.floor(coInRange / coInterval)
  local coIndex = coInRange % coInterval
  local dpIndex = math.floor(coIndex / channel.coPerDp)
  id = channel.firstId + channelIndex * channel.idOffset + dpIndex + 1
  if channel.dpts and channel.dpts[dpIndex + 1] and channel.dpts[dpIndex + 1][1] then
    for _, _dpt in pairs(channel.dpts[dpIndex + 1]) do
      if _dpt[1] == size then
        dpt = _dpt
        break
      end
    end

  end
  return id, dpt[1], dpt[2], dpt[3]
end

-- define a global function call
local func_GetIDandDPTFromCOBySize = function(co, size) return m.GetIDandDPTFromCOBySize(co, size) end
-- overwrite the local function call
function m.SetFunction_GetIDandDPTFromCOBySize(f)
  if type(f)=="function" then
    func_GetIDandDPTFromCOBySize = f
  end
end

-- [global] Get object ID and data point type from communication object number by specific size.
function GetIDandDPTFromCOBySize(co, size)
  return func_GetIDandDPTFromCOBySize(co, size)
end
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- [local] Get communication object number and data point type from object ID.
function m.GetCOandDPTFromID(id, bRead)
  local co = 0xFFFF
  local dpt = m.DPT()
  if id >= device.firstId and id <= device.lastId then
    co = IdTable[id] or co
    dpt = CoTable[co] and CoTable[co][2] or dpt
    return co, dpt[1], dpt[2], dpt[3]
  end

  local channel
  for _, v in pairs(channels) do
    if id >= v.firstId and id <= v.lastId then
      channel = v
      break
    end
  end

  if not channel then
    return co, dpt[1], dpt[2], dpt[3]
  end

  local idInRange = id - channel.firstId
  local channelIndex = math.floor(idInRange / channel.idOffset)
  local dpIndex = idInRange % channel.idOffset - 1
  if dpIndex == -1 then
    -- ID is channel itself
    return co, dpt[1], dpt[2], dpt[3]
  elseif dpIndex >= channel.dpPerChn then
    -- ID is internal data point
    dpIndex = 0
  end
  co = channel.firstCo + channel.coPerDp * (channelIndex * channel.dpPerChn + dpIndex) + (bRead and (channel.coPerDp > 1) and 1 or 0)
  if channel.dpts and channel.dpts[dpIndex + 1] and channel.dpts[dpIndex + 1][1] then
    dpt = channel.dpts[dpIndex + 1][1]
  end
  return co, dpt[1], dpt[2], dpt[3]
end

-- define a global function call
local func_GetCOandDPTFromID = function(id, bRead) return m.GetCOandDPTFromID(id, bRead) end
-- overwrite the local function call
function m.SetFunction_GetCOandDPTFromID(f)
  if type(f)=="function" then
    func_GetCOandDPTFromID = f
  end
end
-- [global] Get object ID and data point type from communication object number.
function GetCOandDPTFromID(id, bRead)
  return func_GetCOandDPTFromID(id, bRead)
end
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- [local] Get communication object number and data point type from object ID with given size or empty if no size DPT found.
function m.GetCOandDPTFromIDBySize(id, bRead, size)
  local co = 0xFFFF
  local dpt = m.DPT()
  if id >= device.firstId and id <= device.lastId then
    co = IdTable[id] or co
    dpt = CoTable[co] and CoTable[co][2] or dpt
    return co, dpt[1], dpt[2], dpt[3]
  end

  local channel
  for _, v in pairs(channels) do
    if id >= v.firstId and id <= v.lastId then
      channel = v
      break
    end
  end

  if not channel then
    return co, dpt[1], dpt[2], dpt[3]
  end

  local idInRange = id - channel.firstId
  local channelIndex = math.floor(idInRange / channel.idOffset)
  local dpIndex = idInRange % channel.idOffset - 1
  if dpIndex == -1 then
    -- ID is channel itself
    return co, dpt[1], dpt[2], dpt[3]
  elseif dpIndex >= channel.dpPerChn then
    -- ID is internal data point
    dpIndex = 0
  end
  co = channel.firstCo + channel.coPerDp * (channelIndex * channel.dpPerChn + dpIndex) + (bRead and (channel.coPerDp > 1) and 1 or 0)

  if channel.dpts and channel.dpts[dpIndex + 1] and channel.dpts[dpIndex + 1][1] then
    for _, _dpt in pairs(channel.dpts[dpIndex + 1]) do
      if _dpt[1] == size then
        dpt = _dpt
        break
      end
    end

  end
  return co, dpt[1], dpt[2], dpt[3]
end

-- define a global function call
local func_GetCOandDPTFromIDBySize = function(id, bRead, size) return m.GetCOandDPTFromIDBySize(id, bRead, size) end
-- overwrite the local function call
function m.SetFunction_GetCOandDPTFromIDBySize(f)
  if type(f)=="function" then
    func_GetCOandDPTFromIDBySize = f
  end
end
-- [global] Get communication object number and data point type from object ID with given size or empty if no size DPT found.
function GetCOandDPTFromIDBySize(id, bRead, size)
  return func_GetCOandDPTFromIDBySize(id, bRead, size)
end
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- URN/CO management

function GetUseUrns()
  return useUrns
end

local urnTable = {}
local coTable = {}

-- helper function to fill the urnTable and coTable from a device-specific channel table
local function FillUrnCoTables(channelTable)
  for _, channel in pairs(channelTable) do
    local count = channel.count or 1
    local offset = channel.offset or #channel.dps

    for i = 1, count do
      local channelName = string.gsub(channel.name, "{0}", i)
      for _, dp in pairs(channel.dps) do
        local urn = channelName .. ":" .. dp.name
        urnTable[urn] = {}

        local tableEntry = { urn, dp.dpt, dp.dpts }

        if (dp.rCo) then
          local rCo = dp.rCo + offset * (i - 1)
          urnTable[urn].rCo = rCo
          coTable[rCo] = tableEntry
        end

        if (dp.wCo) then
          local wCo = dp.wCo + offset * (i - 1)
          urnTable[urn].wCo = wCo
          coTable[wCo] = tableEntry
        end

        if (dp.event == false) then
          urnTable[urn].event = false
        end

        if (dp.internal == false) then
          urnTable[urn].internal = false
        end
      end
    end
  end
end

-- Initialize AppPrg to use with URNs
function m.InitUrn(channelTable)
  useUrns = true
  FillUrnCoTables(channelTable)
end

-- Helper function to get DPT from a coTable entry. If size is given, a matching DPT will be used, if existing
local function GetDptFromTableEntry(co, size)
  if ((not co) or (co == 0xFFFF)) then
    return {0, 0, 0}
  end

  local dpt = coTable[co][2]
  if (not dpt) then
    -- if more than one DPT is set, use the first by default
    local dpts = coTable[co][3]
    dpt = dpts[1]

    -- if a DPT should be selected by size, try that
    -- if no DPT with that size can be found, use default
    if (size and size > 0) then
      for _, val in pairs(dpts) do
        if (val[1] == size) then
          dpt = val
          break
        end
      end
    end
  end
  return dpt or { 0, 0, 0 }
end

function m.GetUrnAndDptFromCo(co, size)
  local coTableEntry = coTable[co]
  if (coTableEntry) then
    local dpt = GetDptFromTableEntry(co, size)
    return coTableEntry[1], dpt[1], dpt[2], dpt[3]
  end
  return "", 0, 0, 0
end

-- define a global function call
local func_GetUrnAndDptFromCo = function(co, size) return m.GetUrnAndDptFromCo(co, size) end
-- overwrite the local function call
function m.SetFunction_GetUrnAndDptFromCo(f)
  if type(f) == "function" then
    func_GetUrnAndDptFromCo = f
  end
end

-- [global] Get object URN and data point type from communication object number by specific size.
function GetUrnAndDptFromCo(co, size)
  return func_GetUrnAndDptFromCo(co, size)
end


function m.GetCoAndDptFromUrn(urn, read, size)
  local urnTableEntry = urnTable[urn]
  if (urnTableEntry) then
    local co
    if (read) then
      co = urnTableEntry.rCo or urnTableEntry.wCo or 0xFFFF
    else
      co = urnTableEntry.wCo or urnTableEntry.rCo or 0xFFFF
    end
    local dpt = GetDptFromTableEntry(co, size)
    return co, dpt[1], dpt[2], dpt[3]
  end
  return 0xFFFF, 0, 0, 0
end

-- define a global function call
local func_GetCoAndDptFromUrn = function(urn, read, size) return m.GetCoAndDptFromUrn(urn, read, size) end
-- overwrite the local function call
function m.SetFunction_GetCOandDPTFromIDBySize(f)
  if type(f) == "function" then
    func_GetCoAndDptFromUrn = f
  end
end
-- [global] Get communication object number and data point type from object ID with given size or empty if no size DPT found.
function GetCoAndDptFromUrn(urn, read, size)
  return func_GetCoAndDptFromUrn(urn, read, size)
end


function EventDpToKnxUrn(urn)
  local tableEntry = urnTable[urn]
  if (tableEntry) then
    if (tableEntry.event == false) then
      return false
    end
    return true
  end
  return false
end

function GdsReadInternalDpUrn(urn)
  local tableEntry = urnTable[urn]
  if (tableEntry) then
    if (tableEntry.internal == false) then
      return false
    end
    return true
  end
  return false
end


-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- helper function to create enums
local __enumID=0;
function m.enum( names )
  local t={}
  for _,k in ipairs(names) do
    t[k]=__enumID
    __enumID = __enumID+1
  end
  return t
end

function m.ValueForKeyInTable(tbl, item)
    for key, value in pairs(tbl) do
        if key == item then return value end
    end
    return 0
end

function m.KeyForValueInTable(tbl, item)
    for key, value in pairs(tbl) do
        if value == item then return key end
    end
    return 0
end

function m.switch(t)
  t.case = function (self,x)
    local f=self[x] or self.default
    if f then
      if type(f)=="function" then
        f(x,self)
      else
        error("case "..tostring(x).." not a function")
      end
    end
  end
  return t
end
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
--[[
ID mangement
]]

m.IdTableDeviceDP  = {}
-- de.gira.schema.channels.GdsDevice
m.IdTableDeviceDP[1] =  1002  -- Ready
m.IdTableDeviceDP[2] =  1003  -- State
m.IdTableDeviceDP[3] =  1004  -- Reset
m.IdTableDeviceDP[4] =  1005  -- Local-Time
m.IdTableDeviceDP[5] =  1006  -- System-Time
m.IdTableDeviceDP[6] =  1007  -- Uptime
-- de.gira.schema.channels.GdsDevice2
m.IdTableDeviceDP[7] =  1008  -- Memory
m.IdTableDeviceDP[8] =  1009  -- Free-Memory
m.IdTableDeviceDP[9] =  1010  -- Storage
m.IdTableDeviceDP[10] = 1011  -- Free-Storage
m.IdTableDeviceDP[11] = 1012  -- Load
m.IdTableDeviceDP[12] = 1013  -- Overload
m.IdTableDeviceDP[13] = 1014  -- Stoppage

function m.InitTableDeviceDP(_IdTableDeviceDP)
  m.IdTableDeviceDP = _IdTableDeviceDP
end

m.IdTableKnxProgrammingModesDP  = {}
-- de.gira.schema.channels.KnxProgrammingModes
m.IdTableKnxProgrammingModesDP[1] =  1051  -- Device1
m.IdTableKnxProgrammingModesDP[2] =  1052  -- Device2
m.IdTableKnxProgrammingModesDP[3] =  1053  -- Device3
m.IdTableKnxProgrammingModesDP[4] =  1054  -- Device4

function m.InitTableProgrammingModesDP(_IdTableKnxProgrammingModesDP)
  m.IdTableKnxProgrammingModesDP = _IdTableKnxProgrammingModesDP
end

m.IdTableKnxStatesDP  = {}
-- de.gira.schema.channels.KnxBusStates
m.IdTableKnxStatesDP[1] =  1061  -- Device1
m.IdTableKnxStatesDP[2] =  1062  -- Device2
m.IdTableKnxStatesDP[3] =  1063  -- Device3
m.IdTableKnxStatesDP[4] =  1064  -- Device4

function m.InitTableKnxStatesDP(_IdTableKnxStatesDP)
  m.IdTableKnxStatesDP = _IdTableKnxStatesDP
end


--[[
Description:
This table contains the data point IDs for write the value to bus on GDS OnValueChanged message.
-- each row is a range of ID's and has the min ID and max ID as value
]]
m.IdTableToBusEvent  = {}
m.IdTableToBusEvent[1] = { m.IdTableDeviceDP[1] , m.IdTableDeviceDP[2] }  -- Ready .. State
m.IdTableToBusEvent[2] = { m.IdTableDeviceDP[7], 1049 }  -- Memory .. OidGdsDeviceChannelLastDp
m.IdTableToBusEvent[3] = { 9000, 9999 }  -- OidDeviceVariableChannelFirstDp .. OidDeviceVariableChannelLastDp

function m.InitTableToBusEvent(_IdTableToBusEvent)
  m.IdTableToBusEvent  = _IdTableToBusEvent
end

-- return true, if found the key
function EventDpToKnx(id)
  for key, value in pairs(m.IdTableToBusEvent) do
    if id >= value[1] and id <= value[2] then
      return true
    end
  end
  return false
end

--[[
Description:
This table contains the data point IDs for set the internal flag on KNX OnGroupValueRead message.
-- each row is a range of ID's and has the min ID and max ID as value
]]
m.IdTableGdsReadInternal  = {}
m.IdTableGdsReadInternal[1] = { m.IdTableDeviceDP[1] , m.IdTableDeviceDP[3] }  -- Ready .. Reset
m.IdTableGdsReadInternal[2] = { m.IdTableDeviceDP[7], m.IdTableKnxStatesDP[1]-1 }  -- Memory .. KnxBusStates device 1
m.IdTableGdsReadInternal[3] = { m.IdTableKnxStatesDP[4]+1, 999999 }  -- > 1064

function m.InitTableGdsReadInternal(_IdTableGdsReadInternal)
  m.IdTableGdsReadInternal  = _IdTableGdsReadInternal
end

-- return true, if found the key
function GdsReadInternalDP(id)
  for key, value in pairs(m.IdTableGdsReadInternal) do
    if id >= value[1] and id <= value[2] then
      return true
    end
  end
  return false
end

--[[
 end ID mangement
]]

-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
--[[
CO management
]]

-- Table of time/date COs and DPTs
-- Timemode server uses wCo to write to KNX, timemode client uses rCo to read from KNX
m.timeCoTable = {
  {
  -- date
    dpt = m.DPT(11, 1),
    wCo = 4,
    rCo = 4
  },
  {
  -- time
    dpt = m.DPT(10, 1),
    wCo = 5,
    rCo = 5
  }
}

function m.InitTimeCoTable(_timeCoTable)
  m.timeCoTable = _timeCoTable
end

function GetTimeCoAndDptByIndex(index, read)
  if (index > #m.timeCoTable) then
    -- index out of bounds
    return 0xFFFF, 0, 0, 0
  end

  local entry = m.timeCoTable[index]
  local co = read and entry.rCo or entry.wCo or 0xFFFF
  local dpt = entry.dpt or {0, 0, 0}
  return co, dpt[1], dpt[2], dpt[3]
end
--[[
end CO management
]]
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
--[[
KNX properties management
]]

-- not supported by default
m.InterfaceObjectId = 0x0000

function m.InitInterfaceObjectId(_InterfaceObjectId)
  m.InterfaceObjectId = _InterfaceObjectId
end

function GetInterfaceObjectId()
  return m.InterfaceObjectId
end

-- no properties by default
-- row = Property ID, type (KnxStack::PropertyDatatype), name
m.PropertyTable = {}

function m.InitPropertyTable(_PropertyTable)
  m.PropertyTable = _PropertyTable
end

function GetPropertyByIndex(_index)
  for key, value in pairs(m.PropertyTable) do
    if key == _index then
      return value[1], value[2], value[3] or 0
    end
  end

  return 0, 0, ""
end

--[[
end KNX properties management
]]
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------

return m
