-- KNX Application Program Handler
-- IDs and COs in this file are specific to the x_tp
--
-- Module with specific functions to x_tp
local x_tp = {}

local m = require("knxappprghandler_base")
local DPT = m.DPT
local Channel = m.Channel

local IdTable = {}
-- de.gira.schema.channels.GdsDevice
IdTable[1002] =  1  -- Ready
IdTable[1003] =  2  -- State
IdTable[1004] =  3  -- Reset
IdTable[1005] =  4  -- Local-Time
IdTable[1007] =  8  -- Uptime
-- de.gira.schema.channels.KnxProgrammingModes
IdTable[1051] = 51  -- Programming-Mode-Device-1
-- de.gira.schema.channels.KnxBusStates
IdTable[1061] = 60  -- Bus-State-Device-1

local CoTable = {}
CoTable[ 1] = { 1002, DPT( 1,  11) }  -- Ready
CoTable[ 2] = { 1003, DPT( 5,  10) }  -- State
CoTable[ 3] = { 1004, DPT( 1,  15) }  -- Reboot
CoTable[ 4] = { 1005, DPT(11,   1) }  -- Date (only as time server)
CoTable[ 5] = { 1005, DPT(10,   1) }  -- Time (only as time server)
CoTable[ 6] = { 1005, DPT(11,   1) }  -- Set Date (only as time client)
CoTable[ 7] = { 1005, DPT(10,   1) }  -- Set Time (only as time client)
CoTable[ 8] = { 1007, DPT(13, 100) }  -- Uptime
CoTable[50] = { 1051, DPT( 1,   1) }  -- Programming Mode
CoTable[51] = { 1051, DPT( 1,   1) }  -- Programming Mode State
CoTable[60] = { 1061, DPT( 1,  11) }  -- Bus State

local device = { firstId = 1002, lastId = 1061, firstCo = 1, lastCo = 60 }

--[[
  firstId  = Range of object IDs for this channel
  firstCo  = Range of communication objects for this channel
  chnCount = Number of channels of this channel type
  dpPerChn = How many data points are in a channel of this type
  coPerDp  = How many communication objects are linked to one data point
  idOffset = Object ID offset between two channels, usually dpPerChn + 1
  chnType  = Name of the channel type
  dpts     = Optional table of data point types for each data point with different size
]]
local channels = {
--  hint: firstId, firstCo, chnCount, dpPerChn, coPerDp, idOffset, chnType, dpts

  -- Switch-Time-Set (50)
  Channel(  12000,    2000,       50,        2,       1,        6, "SwitchTimeSet",
      {
        {DPT( 1, 17)},  -- Triggered
        {DPT(17,  1)}   -- Scene-Index
      }
    ),

  -- Scene-Set (50)
  Channel(  13000,    2500,       50,        1,       1,        3, "SceneSet",
      {
        {DPT(18, 1)}  -- Scene
      }
    ),

  -- Binary-Variable (50)
  Channel(  20000,    5000,       50,        1,       2,        2, "Binary",
      {
        {DPT(1, 2)}  -- boolean
      }
    ),

  -- Unsigned-Variable (50)
  Channel(  20500,    5250,       50,        1,       2,        2, "Unsigned",
      {
        {
          DPT( 12,   1, 32), -- (default) counter pulses (4 Byte)
          DPT(  5,  10,  8), -- counter pulses (1 Byte)
          DPT(  7,   1, 16), -- pulses (2 Byte)
          DPT(232, 600, 24), -- RGB (3 Byte)
        }
      }
    ),

  -- Signed-Variable (50)
  Channel(  21000,    5500,       50,        1,       2,        2, "Signed",
      {
        {
          DPT( 13,   1, 32), -- (default) counter pulses (4 Byte)
          DPT(  6,  10,  8), -- counter pulses (1 Byte)
          DPT(  8,   1, 16), -- pulses  difference(2 Byte)
        }
      }
    ),

  -- Float-Variable (50)
  Channel(  21500,    5750,       50,        1,       2,        2, "Float",
      {
        {
          DPT(  9,   0, 16), -- (default) 2-byte float
          DPT( 14,   0,  32), -- 4-byte float
        }
      }
    ),

  -- Text-Variable (50)
  Channel(  22000,    6000,       50,        1,       2,        2, "Text",
      {
        {
          DPT( 16,   1, 112), -- (default) character String (ISO)
          DPT(  4,   2,   8), -- character (ISO)
        }
      }
    ),

  -- Date-Variable (50)
  Channel(  22500,    6250,       50,        1,       2,        2, "Date" ,
      {
        {
          DPT(11, 1)  -- date
        }
      }
    ),

  -- Time-Variable (50)
  Channel(  23000,    6500,       50,        1,       2,        2, "Time" ,
      {
        {
          DPT(10, 1)  -- time of day
        }
      }
    ),

  -- DateTime-Variable (50)
  Channel(  23500,    6750,       50,        1,       2,        2, "DateTime" ,
      {
        {
          DPT(19, 1)  -- date time
        }
      }
    ),

  -- Percent-Variable (50)
  Channel(  24000,    7000,       50,        1,       2,        2, "Percent" ,
      {
        {
          DPT(5, 1)  -- percentage
        }
      }
    )
}

-- pass references to the device specific tables to the base module functions
m.Init(IdTable, CoTable, device, channels)

-- The global functions accessible from C calls
function x_tp.GetIDandDPTFromCO(co)
  local id;
  local dpt = DPT()
  id, dpt[1], dpt[2], dpt[3] = m.GetIDandDPTFromCO(co)
  -- for SwitchTimeSet map second CO to third DP
  if (co >= 2000) and (co < 2500) and (co % 2 == 1) then
    id = id + 1
  end
  return id, dpt[1], dpt[2], dpt[3]
end

m.SetFunction_GetIDandDPTFromCO(x_tp.GetIDandDPTFromCO)

function x_tp.GetIDandDPTFromCOBySize(co, size)
  local id;
  local dpt = DPT()
  id, dpt[1], dpt[2], dpt[3] = m.GetIDandDPTFromCOBySize(co, size)
  -- for SwitchTimeSet map second CO to third DP
  if (co >= 2000) and (co < 2500) and (co % 2 == 1) then
    id = id + 1
  end
  return id, dpt[1], dpt[2], dpt[3]
end

m.SetFunction_GetIDandDPTFromCOBySize(x_tp.GetIDandDPTFromCOBySize)

function x_tp.GetCOandDPTFromID(id, bRead)
  -- for SwitchTimeSet map first DP to first CO and third DP to second CO, ignore the rest
  if id >= 12000 and id < 13000 then
    local dp = (id - 12000) % 6
    if dp == 3 then
      id = id - 1
    elseif dp ~= 1 then
      id = 0
    end
  end

  return m.GetCOandDPTFromID(id, bRead)
end

m.SetFunction_GetCOandDPTFromID(x_tp.GetCOandDPTFromID)

function x_tp.GetCOandDPTFromIDBySize(id, bRead, size)
  -- for SwitchTimeSet map first DP to first CO and third DP to second CO, ignore the rest
  if id >= 12000 and id < 13000 then
    local dp = (id - 12000) % 6
    if dp == 3 then
      id = id - 1
    elseif dp ~= 1 then
      id = 0
    end
  end

  return m.GetCOandDPTFromIDBySize(id, bRead, size)
end

m.SetFunction_GetCOandDPTFromIDBySize(x_tp.GetCOandDPTFromIDBySize)

-- overwrite global table from package m

-- each row is a range of ID's and has the min ID and max ID as value
local IdTableToBusEvent  = {}
IdTableToBusEvent = m.IdTableToBusEvent
IdTableToBusEvent[1] = { m.IdTableDeviceDP[1] , m.IdTableDeviceDP[2] }  -- Ready .. State
IdTableToBusEvent[2] = { m.IdTableDeviceDP[6], 1061 }  -- Uptime .. Bus State
IdTableToBusEvent[3] = { 1200, 99999 }

m.InitTableToBusEvent(IdTableToBusEvent)

local IdTableGdsReadInternal  = {}
IdTableGdsReadInternal[1] = { m.IdTableDeviceDP[1] , m.IdTableDeviceDP[3] }  -- Ready .. Reset
IdTableGdsReadInternal[2] = { m.IdTableDeviceDP[7], 999999 }  -- up Memory

m.InitTableGdsReadInternal(IdTableGdsReadInternal)

local timeCoTable = {
  {
    dpt = DPT(11, 1),
    wCo = 4,
    rCo = 6
  },
  {
    dpt = DPT(10, 1),
    wCo = 5,
    rCo = 7
  }
}

m.InitTimeCoTable(timeCoTable)

--[[
KNX properties management
]]

-- fix value could disable by set to 0 only
-- see KnxStack::InterfaceObjectType or (own type > 50000 and ID > 50)
local InterfaceObjectId = 0x0003
m.InitInterfaceObjectId(InterfaceObjectId)

-- see KNX specification (for Object type [0..99] the ID > 200)
local PropertyTable = {}
-- row = Property ID, type (KnxStack::PropertyDatatype), name
PropertyTable[1] = {201, 0x02, "TimeMode"} -- PDT_UNSIGNED_CHAR

m.InitPropertyTable(PropertyTable)

--[[
end KNX properties management
]]

return x_tp